<?php
/**
 * This file is part of AgentesComerciales plugin for FacturaScripts
 * Copyright (C) 2025 Your Company <your@email.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

namespace FacturaScripts\Plugins\AgentesComerciales;

use FacturaScripts\Core\Base\AjaxForms\SalesHeaderHTML;
use FacturaScripts\Core\Base\DataBase;
use FacturaScripts\Core\Base\DataBase\DataBaseWhere;
use FacturaScripts\Core\Model\Role;
use FacturaScripts\Core\Model\RoleAccess;
use FacturaScripts\Core\Plugins;
use FacturaScripts\Core\Template\InitClass;
use FacturaScripts\Core\Tools;
use FacturaScripts\Dinamic\Controller\SendTicket;
use FacturaScripts\Dinamic\Lib\ExportManager;
use FacturaScripts\Dinamic\Lib\StockMovementManager;
use FacturaScripts\Dinamic\Lib\Tickets\Service;
use FacturaScripts\Dinamic\Model\AlbaranCliente;
use FacturaScripts\Dinamic\Model\EmailNotification;
use FacturaScripts\Dinamic\Model\FacturaCliente;
use FacturaScripts\Dinamic\Model\PresupuestoCliente;
use FacturaScripts\Dinamic\Model\ReciboCliente;
use FacturaScripts\Dinamic\Model\FormaPago;

/**
 * Description of Init
 *
 * @author Sercicios Guadaira  Miguel Martin  <miguelmartin@servicosguadaira.com>
 */
final class Init extends InitClass
{
    const ROLE_NAME = 'AgentesComerciales';

    public function init(): void
    {
        // extensions
        $this->loadExtension(new Extension\Controller\CopyModel());
        $this->loadExtension(new Extension\Controller\EditCliente());
        $this->loadExtension(new Extension\Controller\EditProducto());

        // tickets
        if (Plugins::isEnabled('Tickets')) {
            SendTicket::addFormat(Service::class, 'ServicioAT', 'service');
        }

        if (Plugins::isEnabled('Randomizer')) {
            $this->loadExtension(new Extension\Controller\Randomizer());
        }

        // export manager
        if (Plugins::isEnabled('PlantillasPDF')) {
            ExportManager::addOptionModel('PlantillasPDFserviciosExport', 'PDF', 'ServicioAT');
        } else {
            ExportManager::addOptionModel('PDFserviciosExport', 'PDF', 'ServicioAT');
        }

        // mod para los documentos de venta
        SalesHeaderHTML::addMod(new Mod\SalesHeaderHTMLMod());

        // mod y extensión para StockAvanzado
        if (Plugins::isEnabled('StockAvanzado')) {
            StockMovementManager::addMod(new Mod\StockMovementMod());
            $this->loadExtension(new Extension\Model\TrabajoAT());
        }

        // NUEVO: Inicializar eventos para fraccionamiento de recibos
        $this->initReceiptEvents();
    }

    public function uninstall(): void
    {
    }

    public function update(): void
    {
        $this->fixMissingAgents();
        $this->fixMissingCustomers();

        new Model\EstadoAT();
        new Model\MaquinaAT();
        new Model\PrioridadAT();
        new Model\TipoAT();
        new Model\ServicioAT();
        new PresupuestoCliente();
        new AlbaranCliente();
        new FacturaCliente();

        $this->setupSettings();
        $this->createRoleForPlugin();
        $this->updateEmailNotifications();
    }

    /**
     * Inicializa los eventos para el fraccionamiento de recibos
     */
    private function initReceiptEvents(): void
    {
        Tools::log()->info('INIT: Sistema de fraccionamiento inicializado');
        
        // El fraccionamiento se manejará mediante una extensión del modelo
        $this->loadExtension(new Extension\Model\FacturaCliente());
    }

    /**
     * Procesa un recibo después de guardarlo para ver si necesita fraccionamiento
     */
    private function procesarReciboDespuesDeGuardar($recibo): void
    {
        Tools::log()->info('EVENTO RECIBO: Procesando recibo guardado: ' . $recibo->numero);
        
        if ($this->necesitaFraccionamientoRecibo($recibo)) {
            Tools::log()->info('EVENTO RECIBO: Fraccionando recibo para factura: ' . $recibo->codigofactura);
            
            // Eliminar el recibo original
            $recibo->delete();
            
            // Crear los recibos fraccionados
            $this->crearRecibosFraccionadosEvento($recibo);
        }
    }
    
    /**
     * Verifica si un recibo necesita fraccionamiento
     */
    private function necesitaFraccionamientoRecibo($recibo): bool
    {
        Tools::log()->info('EVENTO RECIBO: Verificando fraccionamiento...');
        
        // Solo para recibos no pagados
        if ($recibo->pagado) {
            Tools::log()->info('EVENTO RECIBO: Recibo ya pagado, no fraccionar');
            return false;
        }

        // Obtener la forma de pago
        $formaPago = new FormaPago();
        if (!$formaPago->loadFromCode($recibo->codpago)) {
            Tools::log()->info('EVENTO RECIBO: No se pudo cargar forma de pago: ' . $recibo->codpago);
            return false;
        }

        $descripcion = strtolower($formaPago->descripcion);
        Tools::log()->info('EVENTO RECIBO: Descripción forma de pago: ' . $descripcion);
        
        // Verificar si es fraccionamiento
        $esFraccionamiento = (strpos($descripcion, '30,60,90') !== false || 
                             strpos($descripcion, '30-60-90') !== false ||
                             strpos($descripcion, '30,60') !== false || 
                             strpos($descripcion, '30-60') !== false);

        Tools::log()->info('EVENTO RECIBO: Es fraccionamiento? ' . ($esFraccionamiento ? 'SI' : 'NO'));

        return $esFraccionamiento;
    }
    
    /**
     * Crea los recibos fraccionados usando eventos
     */
    private function crearRecibosFraccionadosEvento($reciboOriginal): void
    {
        // Obtener la forma de pago
        $formaPago = new FormaPago();
        if (!$formaPago->loadFromCode($reciboOriginal->codpago)) {
            return;
        }

        $descripcion = strtolower($formaPago->descripcion);
        
        // Determinar los días de vencimiento
        $diasVencimiento = [];
        if (strpos($descripcion, '30,60,90') !== false || strpos($descripcion, '30-60-90') !== false) {
            $diasVencimiento = [30, 60, 90];
        } elseif (strpos($descripcion, '30,60') !== false || strpos($descripcion, '30-60') !== false) {
            $diasVencimiento = [30, 60];
        }

        if (empty($diasVencimiento)) {
            return;
        }

        $numRecibos = count($diasVencimiento);
        $importeTotal = $reciboOriginal->importe;
        
        // Calcular importes fraccionados
        $importeRecibo = round($importeTotal / $numRecibos, 2);
        $importeUltimo = $importeTotal - ($importeRecibo * ($numRecibos - 1));

        Tools::log()->info("EVENTO RECIBO: Creando {$numRecibos} recibos fraccionados");

        foreach ($diasVencimiento as $index => $dias) {
            // Crear nuevo recibo
            $nuevoRecibo = new ReciboCliente();
            
            // Copiar datos del recibo original (excluyendo el ID)
            $datosOriginales = $reciboOriginal->toArray();
            unset($datosOriginales['idrecibo']);
            $nuevoRecibo->loadFromData($datosOriginales);
            
            // Asignar importe correspondiente
            $nuevoRecibo->importe = ($index === $numRecibos - 1) ? $importeUltimo : $importeRecibo;
            
            // Calcular fecha de vencimiento - CORREGIDO
            $fechaVencimiento = new \DateTime($reciboOriginal->fecha);
            $intervalo = "P{$dias}D";
            $fechaVencimiento->add(new \DateInterval($intervalo));
            $nuevoRecibo->vencimiento = $fechaVencimiento->format('Y-m-d');
            
            // Obtener número correlativo único
            $nuevoRecibo->numero = $this->getNextReciboNumberEvento($reciboOriginal->idempresa);
            
            // Estado inicial
            $nuevoRecibo->pagado = false;
            $nuevoRecibo->vencido = $this->estaVencidaEvento($nuevoRecibo->vencimiento);
            $nuevoRecibo->liquidado = 0;
            $nuevoRecibo->gastos = 0;
            
            // Observaciones descriptivas
            $nuevoRecibo->observaciones = "Recibo " . ($index + 1) . "/{$numRecibos} - Vencimiento a {$dias} días (" . $fechaVencimiento->format('d/m/Y') . ")";

            // Guardar el recibo fraccionado
            if ($nuevoRecibo->save()) {
                Tools::log()->info("EVENTO RECIBO: Creado recibo fraccionado: {$nuevoRecibo->numero} por {$nuevoRecibo->importe}€ a {$dias} días");
            } else {
                Tools::log()->error("EVENTO RECIBO: Error creando recibo fraccionado " . ($index + 1) . "/{$numRecibos} a {$dias} días");
            }
        }
    }
    
    /**
     * Obtiene el siguiente número correlativo de recibo
     */
    private function getNextReciboNumberEvento(int $idempresa): int
    {
        $recibo = new ReciboCliente();
        $sql = "SELECT COALESCE(MAX(numero), 0) + 1 as siguiente FROM " . $recibo->tableName() . " WHERE idempresa = " . $idempresa;
             
        $db = new DataBase();
        $data = $db->select($sql);
        return empty($data) ? 1 : (int) $data[0]['siguiente'];
    }

    /**
     * Verifica si una fecha está vencida
     */
    private function estaVencidaEvento(string $fechaVencimiento): bool
    {
        $hoy = new \DateTime();
        $vencimiento = new \DateTime($fechaVencimiento);
        
        return $vencimiento < $hoy;
    }

    private function createRoleForPlugin(): void
    {
        $dataBase = new DataBase();
        $dataBase->beginTransaction();

        // creates the role if not exists
        $role = new Role();
        if (false === $role->loadFromCode(self::ROLE_NAME)) {
            $role->codrole = $role->descripcion = self::ROLE_NAME;
            if (false === $role->save()) {
                // rollback and exit on fail
                $dataBase->rollback();
                return;
            }
        }

        // checks the role permissions
        $nameControllers = ['EditMaquinaAT', 'EditServicioAT', 'ListServicioAT', 'NewServicioAT'];
        foreach ($nameControllers as $nameController) {
            $roleAccess = new RoleAccess();
            $where = [
                new DataBaseWhere('codrole', self::ROLE_NAME),
                new DataBaseWhere('pagename', $nameController)
            ];
            if ($roleAccess->loadFromCode('', $where)) {
                // permission exists? Then skip
                continue;
            }

            // creates the permission if not exists
            $roleAccess->allowdelete = true;
            $roleAccess->allowupdate = true;
            $roleAccess->codrole = self::ROLE_NAME;
            $roleAccess->pagename = $nameController;
            $roleAccess->onlyownerdata = false;
            if (false === $roleAccess->save()) {
                // rollback and exit on fail
                $dataBase->rollback();
                return;
            }
        }

        // without problems = Commit
        $dataBase->commit();
    }

    private function fixMissingAgents(): void
    {
        // si no existe la tabla, no hacemos nada
        $db = new DataBase();
        foreach (['serviciosat', 'serviciosat_trabajos'] as $table) {
            if (false === $db->tableExists($table)) {
                break;
            }

            $sql = 'UPDATE ' . $table . ' SET codagente = NULL WHERE codagente IS NOT NULL AND codagente NOT IN (SELECT codagente FROM agentes);';
            $db->exec($sql);
        }
    }

    private function fixMissingCustomers(): void
    {
        // si no existe la tabla, no hacemos nada
        $db = new DataBase();
        if (false === $db->tableExists('serviciosat')) {
            return;
        }

        $db = new DataBase();
        $sql = 'UPDATE serviciosat SET codcliente = NULL WHERE codcliente IS NOT NULL AND codcliente NOT IN (SELECT codcliente FROM clientes);';
        $db->exec($sql);
    }

    private function setupSettings(): void
    {
        $defaults = [
            'print_ticket_footer_text' => '',
            'print_pdf_footer_text' => '',
            'longnumero' => 6,
            'patron' => 'COM{ANYO}-{NUM}', // Cambiado de SER a COM (Comisiones)
            'workstatus' => 1
        ];

        foreach ($defaults as $key => $value) {
            Tools::settings('agentescomerciales', $key, $value); // Cambiado el settings key
        }
        Tools::settingsSave();
    }

    private function updateEmailNotifications(): void
    {
        $notificationModel = new EmailNotification();
        $keys = [
            'new-service-assignee', 'new-service-agent', 'new-service-customer',
            'new-service-status', 'new-service-user'
        ];
        foreach ($keys as $key) {
            if ($notificationModel->loadFromCode($key)) {
                continue;
            }

            $notificationModel->name = $key;
            $notificationModel->body = Tools::lang()->trans($key . '-body');
            $notificationModel->subject = Tools::lang()->trans($key);
            $notificationModel->enabled = false;
            $notificationModel->save();
        }
    }
}